const { PSKEYS } = require('./consts');
const util = require('util');
const { StringDecoder } = require('string_decoder');

function swap_keyvalue(keyvalues) {
    var ret = {};
    for (let key in keyvalues) {
        ret[keyvalues[key]] = key;
    }
    return ret;
}

function _makeSymmetricalConverter(valueMapping) {
    function converter(fromUI, value) {
        let mapping = valueMapping;
        if (!fromUI) {
            mapping = swap_keyvalue(mapping);
        }

        if (typeof value == 'string') {
            value = value.toLowerCase();
        }

        return mapping[value];
    }

    return converter;
}

function makeVolumeConverter(scale) {
    function _volumeConverter(fromUI, value) {
        const MAX_PSKEY_VOLUME = scale;

        // While the UI presets a scale of 0-100, 10% being the minimum, the scale is actually 0-90, where 10% == 0.
        // YACHH
        if (fromUI) {
            value = (value - 10).toString();
        }

        if (fromUI) {
            value = parseInt(value);
            return Math.round((MAX_PSKEY_VOLUME * value) / 90);
        } else {
            return Math.round((90 * value) / MAX_PSKEY_VOLUME) + 10;
        }
    }

    return _volumeConverter;
}

const GENERIC_VOLUME_SCALE = 9;
const LARGE_VOLUME_SCALE = 127;
const _genericVolumeConverter = makeVolumeConverter(GENERIC_VOLUME_SCALE);
const _largeVolumeConverterWithScale = makeVolumeConverter(LARGE_VOLUME_SCALE);

function _largeVolumeConverter(fromUI, value) {
    // According to Oleh. These are the correct value mappings. They do not seem to accurately
    // reflect the proper mathematical scaling of the 0-100 value to 0-127, so I have hardcoded them according to Oleh's provided table
    if (fromUI) {
        value = value - 10;
    }

    let hardcodedMapping = {
        '0': 0,
        '10': 14,
        '20': 28,
        '30': 42,
        '40': 56,
        '50': 70,
        '60': 84,
        '70': 98,
        '80': 112,
        '90': 126
    };

    if (!fromUI) {
        hardcodedMapping = swap_keyvalue(hardcodedMapping);
    }

    value = value.toString();
    let newValue = hardcodedMapping[value];

    // If the hardcoded mapping yielded nothing, then return the calculated value
    if (newValue === undefined) {
        if (fromUI) {
            value = parseInt(value) + 10;
        }
        return _largeVolumeConverterWithScale(fromUI, value);
    } else {
        let ret = fromUI ? newValue : (parseInt(newValue) + 10).toString();
        return ret;
    }
}

// Values taken from BTM/VM/Cardo/CAIP/caip_parser_v_64.c
const _radioBandConverter = _makeSymmetricalConverter({
    worldwide: 0,
    japan: 1
});

// Values taken from BTM/VM/Cardo/VoicePrompts/cardo_voice_prompts_mngr.h:vcm_lang_t
const _languageConverter = _makeSymmetricalConverter({
    english: 0,
    english_uk: 1,
    spanish: 2,
    french: 3,
    german: 4,
    italian: 5,
    portuguese: 6,
    dutch: 7,
    russian: 8,
    japanese: 9,
    chinese: 10,
    hebrew: 11,
    korean: 12
});

const _thresholdConverter = _makeSymmetricalConverter({
    low: 0,
    medium: 1,
    high: 2
});

const _batteryTypeConverter = _makeSymmetricalConverter({
    'li-on': 0,
    '2xaa': 1,
    '3xaa': 2,
    '3xaaa': 3
});

const _speakerSwitchConverter = _makeSymmetricalConverter({
    'left-right': 0,
    'right-left': 1
});

const _chargeLEDLocationConverter = _makeSymmetricalConverter({
    pro: 0,
    'power board': 1
});

const _magicWordConverter = _makeSymmetricalConverter({
    'hey cardo': 0,
    'hey sonis': 1
});

function _intercomChannelConverter(fromUI, value) {
    if (fromUI) {
        return (value - 1) * 2;
    } else {
        return (value + 2) / 2;
    }
}

const _pskeyConverters = {
    [PSKEYS.FMBand]: _radioBandConverter,
    [PSKEYS.language]: _languageConverter,
    [PSKEYS.VAD]: _thresholdConverter,
    [PSKEYS.micSensitivity]: _thresholdConverter,
    [PSKEYS.LDVolume]: _genericVolumeConverter,
    [PSKEYS.selfHearingVolume]: _genericVolumeConverter,
    [PSKEYS.secondaryAudioVolume]: _largeVolumeConverter,
    [PSKEYS.batteryType]: _batteryTypeConverter,
    [PSKEYS.speakerSwitch]: _speakerSwitchConverter,
    [PSKEYS.chargeLEDLocation]: _chargeLEDLocationConverter,
    [PSKEYS.magicWord]: _magicWordConverter
};

function convertToPSKeyValue(key, value) {
    let converter = function(_, value) {
        console.log(typeof value);
        console.log(value);
        if (typeof value === 'boolean') {
            return value ? 1 : 0;
        }

        if (typeof value == 'number') {
            return value;
        }

        if (typeof value == 'string') {
            return value;
        }

        if (value instanceof Array) {
            return value;
        }

        let errorMessage = util.format(
            'Cannot convert value of type=%s and value=%s to pskey value',
            typeof value,
            value
        );
        console.log(errorMessage);
        throw new Error(util.format());
    };

    if (_pskeyConverters[key] !== undefined) {
        converter = _pskeyConverters[key];
    }

    return converter(true /* fromUI */, value);
}

function convertFromPSKeyValue(key, value) {
    let converter = function(_, value) {
        console.log('Converting from buffer ' + key + 'value: ' + value);
        if (value instanceof Buffer) {
            const decoder = new StringDecoder('ascii');
            let nullIndex = value.indexOf(0);
            if (nullIndex != -1) {
                return decoder.write(value.slice(0, nullIndex));
            } else {
                return decoder.write(value);
            }
        }

        return value;
    };

    if (_pskeyConverters[key] !== undefined) {
        console.log('Found custom converter for ' + key);
        converter = _pskeyConverters[key];
    }

    console.log('Converting from ' + key + ' value: ' + value);
    value = converter(false, value);
    console.log('Converted value: ' + value);
    return value;
}

module.exports = {
    convertToPSKeyValue: convertToPSKeyValue,
    convertFromPSKeyValue: convertFromPSKeyValue
};
